// /chat/public/assets/js/main.js

document.addEventListener('DOMContentLoaded', () => {
  // Seletores do DOM
  const chatListEl         = document.getElementById('chat-list');
  const chatMessagesEl     = document.getElementById('chat-messages');
  const chatInputEl        = document.getElementById('chat-input');
  const sendBtn            = document.getElementById('send-btn');
  const newChatLinks       = document.querySelectorAll('.create-chat-link');
  const currentChatTitle   = document.getElementById('current-chat-title');
  const modelSelectEl      = document.getElementById('model-select');
  const renameChatBtn      = document.getElementById('rename-chat-btn');
  const deleteChatBtn      = document.getElementById('delete-chat-btn');

  let currentConversationId = null;
  let currentModel          = modelSelectEl.value || 'gpt-3.5-turbo';

  // —————————————————————————————————————————
  // 1) Remova qualquer lógica de tema deste arquivo.
  //    Todo dark/light ficará no header.php.
  // —————————————————————————————————————————

  // —————————————————————————————————————————
  // 2) Alteração de modelo (select)
  // —————————————————————————————————————————
  modelSelectEl.addEventListener('change', () => {
    currentModel = modelSelectEl.value;
    if (currentConversationId) {
      fetch('update_conversation_model.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          conversation_id: currentConversationId,
          new_model:       currentModel
        })
      });
    }
  });

  // —————————————————————————————————————————
  // 3) “Nova Conversa” (vários links com .create-chat-link)
  // —————————————————————————————————————————
  newChatLinks.forEach(link => {
    link.addEventListener('click', e => {
      e.preventDefault();
      currentConversationId = null;
      currentChatTitle.textContent = 'Nova Conversa';
      chatMessagesEl.innerHTML = '<p class="placeholder">Envie uma mensagem para começar.</p>';
      chatInputEl.value = '';
      modelSelectEl.value = 'gpt-3.5-turbo';
      currentModel = 'gpt-3.5-turbo';
      renameChatBtn.disabled = true;
      deleteChatBtn.disabled = true;
      document.querySelectorAll('.chat-item.active').forEach(el => el.classList.remove('active'));
    });
  });

  // —————————————————————————————————————————
  // 4) Função que envia mensagem ao servidor
  // —————————————————————————————————————————
  async function sendMessageToServer(text) {
    const payload = {
      conversation_id: currentConversationId || 0,
      message:         text,
      model:           currentModel
    };

    try {
      const res = await fetch('chat_handler.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
      });
      if (!res.ok) {
        let errText = `HTTP ${res.status}`;
        try {
          const errJson = await res.json();
          if (errJson.error) errText = errJson.error;
        } catch {}
        throw new Error(errText);
      }
      return await res.json();
    } catch (err) {
      console.error(err);
      return { error: err.message };
    }
  }

  // —————————————————————————————————————————
  // 5) Envia a mensagem ao clicar em “Enviar”
  //    → limpa o campo antes de chamar a API
  // —————————————————————————————————————————
  sendBtn.addEventListener('click', async () => {
    const text = chatInputEl.value.trim();
    if (!text) return;

    // Limpa imediatamente o campo de entrada
    chatInputEl.value = '';

    // Exibe a mensagem do usuário
    appendUserMessage(text);

    // Exibe indicador de “digitando...”
    const loadingIndicator = appendLoadingIndicator();

    // Chama o servidor
    const result = await sendMessageToServer(text);

    // Remove indicador
    removeLoadingIndicator();

    if (result.error) {
      appendAssistantMessage('❗ Erro: ' + result.error);
      return;
    }

    // Se for conversa nova, atualiza ID e recarrega lista
    if (!currentConversationId && result.conversation_id) {
      currentConversationId = result.conversation_id;
      await loadChatList();
      const newItem = chatListEl.querySelector(`[data-id='${currentConversationId}']`);
      if (newItem) newItem.classList.add('active');
      renameChatBtn.disabled = false;
      deleteChatBtn.disabled = false;
      currentChatTitle.textContent = newItem ? newItem.textContent : 'Chat';
    }

    appendAssistantMessage(result.assistant_response);
  });

  // —————————————————————————————————————————
  // 6) Funções auxiliares para inserir mensagens no DOM
  // —————————————————————————————————————————
  function appendUserMessage(text) {
    if (chatMessagesEl.querySelector('.placeholder')) {
      chatMessagesEl.innerHTML = '';
    }
    const userMsg = document.createElement('div');
    userMsg.classList.add('message', 'user');
    userMsg.textContent = text;
    chatMessagesEl.appendChild(userMsg);
    chatMessagesEl.scrollTop = chatMessagesEl.scrollHeight;
  }

  function appendAssistantMessage(text) {
    const assistantMsg = document.createElement('div');
    assistantMsg.classList.add('message', 'assistant');
    assistantMsg.textContent = text;
    chatMessagesEl.appendChild(assistantMsg);
    chatMessagesEl.scrollTop = chatMessagesEl.scrollHeight;
  }

  function appendLoadingIndicator() {
    const loadingIndicator = document.createElement('div');
    loadingIndicator.classList.add('message', 'assistant');
    loadingIndicator.textContent = 'O assistente está digitando…';
    loadingIndicator.setAttribute('id', 'loading-indicator');
    chatMessagesEl.appendChild(loadingIndicator);
    chatMessagesEl.scrollTop = chatMessagesEl.scrollHeight;
    return loadingIndicator;
  }

  function removeLoadingIndicator() {
    const loadingEl = document.getElementById('loading-indicator');
    if (loadingEl) loadingEl.remove();
  }

  // —————————————————————————————————————————
  // 7) Carrega a lista de conversas do usuário (chat_list.php)
  // —————————————————————————————————————————
  async function loadChatList() {
    try {
      const res = await fetch('chat_list.php');
      if (!res.ok) throw new Error('Falha ao carregar conversas');
      const conversas = await res.json();
      chatListEl.innerHTML = '';

      if (conversas.length === 0) {
        chatListEl.innerHTML = '<p class="placeholder">Nenhuma conversa ainda.</p>';
        return;
      }

      conversas.forEach(item => {
        const div = document.createElement('div');
        div.classList.add('chat-item');
        div.textContent = item.titulo;
        div.dataset.id = item.id;
        if (parseInt(item.id) === parseInt(currentConversationId)) {
          div.classList.add('active');
        }
        div.addEventListener('click', () => {
          selectConversation(item.id, item.titulo);
        });
        chatListEl.appendChild(div);
      });
    } catch (err) {
      console.error(err);
      chatListEl.innerHTML = '<p class="placeholder">Erro ao carregar conversas.</p>';
    }
  }

  // —————————————————————————————————————————
  // 8) Seleciona uma conversa (carrega histórico e modelo)
  // —————————————————————————————————————————
  function selectConversation(id, title) {
    currentConversationId = id;
    currentChatTitle.textContent = title;

    document.querySelectorAll('.chat-item.active').forEach(el => el.classList.remove('active'));
    const selectedEl = chatListEl.querySelector(`[data-id='${id}']`);
    if (selectedEl) selectedEl.classList.add('active');

    renameChatBtn.disabled = false;
    deleteChatBtn.disabled = false;

    chatMessagesEl.innerHTML = '<p class="placeholder">Carregando mensagens…</p>';
    fetch(`chat_history.php?conversation_id=${id}`)
      .then(res => res.json())
      .then(msgs => {
        chatMessagesEl.innerHTML = '';
        // Busca e aplica o modelo desta conversa
        fetch(`get_conversation_model.php?conversation_id=${id}`)
          .then(r => r.json())
          .then(data => {
            if (data.modelo) {
              modelSelectEl.value = data.modelo;
              currentModel = data.modelo;
            }
          });

        if (!msgs.length) {
          chatMessagesEl.innerHTML = '<p class="placeholder">Sem mensagens ainda.</p>';
          return;
        }
        msgs.forEach(m => {
          const div = document.createElement('div');
          div.classList.add('message', m.role === 'user' ? 'user' : 'assistant');
          div.textContent = m.content;
          chatMessagesEl.appendChild(div);
        });
        chatMessagesEl.scrollTop = chatMessagesEl.scrollHeight;
      })
      .catch(err => {
        console.error(err);
        chatMessagesEl.innerHTML = '<p class="placeholder">Erro ao carregar mensagens.</p>';
      });
  }

  // —————————————————————————————————————————
  // 9) Botões “Renomear” e “Excluir”
  // —————————————————————————————————————————
  renameChatBtn.disabled = true;
  deleteChatBtn.disabled = true;

  renameChatBtn.addEventListener('click', () => {
    if (!currentConversationId) return;
    const novoTitulo = prompt('Digite o novo título para esta conversa:', currentChatTitle.textContent);
    if (novoTitulo && novoTitulo.trim()) {
      fetch('rename_conversation.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          conversation_id: currentConversationId,
          new_title:       novoTitulo.trim()
        })
      })
        .then(res => res.json())
        .then(data => {
          if (data.success) {
            currentChatTitle.textContent = novoTitulo.trim();
            const item = chatListEl.querySelector(`[data-id='${currentConversationId}']`);
            if (item) item.textContent = novoTitulo.trim();
          } else {
            alert('Erro ao renomear: ' + (data.error || ''));
          }
        });
    }
  });

  deleteChatBtn.addEventListener('click', () => {
    if (!currentConversationId) return;
    if (!confirm('Deseja realmente excluir esta conversa?')) return;
    fetch('delete_conversation.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ conversation_id: currentConversationId })
    })
      .then(res => res.json())
      .then(data => {
        if (data.success) {
          const item = chatListEl.querySelector(`[data-id='${currentConversationId}']`);
          if (item) item.remove();

          currentConversationId = null;
          currentChatTitle.textContent = 'Nova Conversa';
          chatMessagesEl.innerHTML = '<p class="placeholder">Envie uma mensagem para começar.</p>';
          renameChatBtn.disabled = true;
          deleteChatBtn.disabled = true;
        } else {
          alert('Erro ao excluir: ' + (data.error || ''));
        }
      });
  });

  // —————————————————————————————————————————
  // 10) Chama loadChatList quando a página abre
  // —————————————————————————————————————————
  loadChatList();

  // —————————————————————————————————————————
  // 11) ENTER envia mensagem, SHIFT+ENTER pula linha
  // —————————————————————————————————————————
  chatInputEl.addEventListener('keydown', e => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendBtn.click();
    }
    // SHIFT+ENTER já deixa o <textarea> pular linha por padrão
  });
});
